#include "sift.h"


void g2d(float* target, 
         unsigned char* src,
         int w,
         int h,
         float sigma);

/****************************************
 * buildSS:
 * Build Scale Space by Gaussian
 *
 * Parameter
 * octaves: input, generated by buildOB
 * scaleSpace: output, total OxS images
 * O: input, number of octaves
 * S: input, scales per octave
 * octavesW: output, width of each octave
 * octavesH: output, height of each octave
 * sigmas: input, sigma for each scale
 ***************************************/

void buildSS(unsigned char* octaves[MAX_O], 
             float* scaleSpace[MAX_O][MAX_S], 
             int O, int S, 
             int* octavesW, int* octavesH, 
             float sigmas[MAX_O][MAX_S]){

    // common variables
    int i, j;
    for(i = 0; i < O; i++){
        for(j = 0; j < S; j++){
            // 2D Gaussian
            g2d(scaleSpace[i][j], 
                octaves[i],
                octavesW[i],
                octavesH[i],
                sigmas[i][j]);
        }
    }
  
}

/********************************************
 * g2d
 * 2D gaussian filter
 * implemented as two 1D gaussian filter
 * one horizontal, one vertical
 *
 * Parameters
 * target: output, filtered by 2D gaussian
 * src: input, source image
 * w: width of the image (same for target and src)
 * h: height of the image (same for target and src)
 * sigma: sigma of gaussian kernel
 *******************************************/
void g2d(float* target, 
         unsigned char* src,
         int w,
         int h,
         float sigma){
  
    // common variables
    int i, j;
    float tmp;
    float norm = 0.0f; // variable for normalization

    /*******************************
     * gR:
     * radius of the gaussian window
     * gW:
     * width of the gaussian window
     * gW = gR * 2 + 1
     * if gR == 2, the gW == 5
     * which is a 5x5 window
     *******************************/
    int gR;
    int gW;

    // 1D gaussian coeficients
    float* coef1d;

    // pad the source image with boarders,
    // which has the width of gR
    unsigned char* srcPadded;

    // intermediate 1D gaussian result
    float* intermediate;

    // derive gR and gW
    if(sigma*4.0f > 1.0f){
        gR = (int)(sigma * 4.0f);
    }else{
        gR = 1;
    }
    gW = gR * 2 + 1;
  
    // allocate memory for gaussian coefficients
    coef1d = (float *)malloc( gW * sizeof(float) );

    // allocat ememory for padded source image
    // srcPadded has the type of (unsigned char)
    srcPadded = 
        (unsigned char *)
        malloc(
            (w+gR*2)*(h+gR*2)*
            sizeof(unsigned char));

    // Allocat ememory for intermediate image.
    // Intermediate has the same size of srcPadded.
    // intermediate has the type of (float)
    intermediate = 
        (float *)
        malloc((w+gR*2)*(h+gR*2)*sizeof(float));

    /****************************************
     * Compute Gaussian Coefficients
     ***************************************/

    norm = 0.0f;
    for(i = 0; i < gW; i++){
        tmp = (float)((float)i - (float)gR) / sigma;
        coef1d[i] = exp( -1.0f * tmp * tmp / 2.0f );
        norm = norm + coef1d[i];
    }
    for(i = 0; i < gW; i++){
        coef1d[i] = coef1d[i] / norm;
    }


    /****************************************
     * Padding "0" and copy src to srcPadded
     * Padding "0" to intermediate
     * Input 1 : src, size (w)x(h)
     * Input 2 : intermediate, size (w+gR*2)x(h+gR*2)
     * Output 1 : srcPadded, size (w+gR*2)x(h+gR*2)
     * Output 2 : intermediate, size (w+gR*2)x(h+gR*2)
     ***************************************/
    for(i = 0; i < h+gR*2; i++){
        for(j = 0; j < w+gR*2; j++){
            if( i < gR || i > h+gR-1 
                || j < gR || j > w+gR-1 ){
                // padding srcPadded
                srcPadded[i*(w+gR*2)+j] = 0;
                // padding intermediate
                intermediate[i*(w+gR*2)+j] = 0.0f;
            }else{
                // copy the src to srcPadded
                srcPadded[i*(w+gR*2)+j] = 
                    src[(i-gR)*w+(j-gR)];
            }
        }
    }

    /**********************************************
     * Horizontal 1D Gaussian
     * Input: srcPadded, size: (w+gR*2)x(h+gR*2)
     * Output: intermediate, size: (w+gR*2)x(h+gR*2)
     *********************************************/
    for(i = gR; i < h+gR; i++){
        for(j = gR; j < w+gR; j++){
            float accu = 0.0f;
            int ii;
            for(ii = (-1 * gR); ii < (gR + 1); ii++){
                // convolute with gaussian window
                // note: srcPadded is unsigned char
                accu += 
                    coef1d[ii + gR] * 
                    (float)srcPadded[i*(w+gR*2)+j+ii];
            }
            intermediate[i*(w+gR*2)+j] = accu;
        }
    }

    /**********************************************
     * Vertical 1D Gaussian
     * Input: intermediate, size: (w+gR*2)x(h+gR*2)
     * Output: target, size: (w)x(h)
     *********************************************/
    for(i = gR; i < h+gR; i++){
        for(j = gR; j < w+gR; j++){
            float accu = 0.0f;
            int ii;
            for(ii = (-1 * gR); ii < (gR + 1); ii++){
                // convolute with gaussian window
                accu += 
                    coef1d[ii + gR] * 
                    (float)
                    intermediate[(i+ii)*(w+gR*2)+j];
            }
            target[(i-gR)*w+(j-gR)] = accu;
        }
    }

    /**********************************************
     * Free the allocated memory
     * Input: intermediate, size: (w+gR*2)x(h+gR*2)
     * Output: target, size: (w)x(h)
     *********************************************/
    free(coef1d);
    free(srcPadded);
    free(intermediate);

}
