#include "sift.h"

#define CURVATURE_THRESHOLD 10.0f

void checkExtrema(float* self,
                  float* up,
                  float* down,
                  int w,
                  int h,
                  int r,
                  int c,
                  float peakThres,
                  int* flagExtrema);

void checkEdge(float *dogImg, 
               int w,
               int h,
               int r,
               int c,
               int *flagEdge);

/************************************************
 * extreme:
 * Find extreme points, store them in linked list
 *
 * Parameter
 * keyHead: output, head of the linked list
 * keyN: output, nr. of key points in linked list
 * dog: input, generated by DoG (diff of gaussian)
 * O: number of octaves
 * S: scales per octave
 * octavesW: output, width of each octave
 * octavesH: output, height of each octave
 ***********************************************/

void extreme(pointList** keyHead, int* keyN,
             float* dog[MAX_O][MAX_S-1], 
             int O, int S,
             int* octavesW, int* octavesH){

    // common variables
    int i, j, ii ,jj;
    float* self;
    float* up;
    float* down;
    int w, h;
    // peak threshold (emperical number)
    float peakThres = 3.0;
    // flag for extrema
    int flagExtrema;
    // flag for edge
    int flagEdge;

    /********************************************
     * Patch:
     * Elimiate feature points near the border,
     * such that the orientation stage will not
     * have access outside the array bound.
     * Sigma should have been passed from "sift.c",
     * but it is re-calculated here to avoid
     * changing the interface of the functions.
     *******************************************/
    // sigma at each scale
    float sigma0 = 1.6f;
    float sigmas[MAX_O][MAX_S];
    // radius of orientation window
    float radius;
    // gaussian sigmas at each scale
    for(i = 0; i < O; i++){
        for(j = 0; j < S; j++){
            sigmas[i][j] = 
                sigma0 * 
                pow(sqrt(2.0f), (float)j/(float)(S-3));
        }
    }

  
    for(i = 0; i < O; i++){
        for(j = 1; j < S-2; j++){
            w = octavesW[i];
            h = octavesH[i];
            self = dog[i][j];
            up = dog[i][j+1];
            down = dog[i][j-1];
            /**********************************
             * Patch:
             * Avoid points close to the border.
             * Loop bounds are changed accordingly.
             * The margin are set to "radius",
             * which is the radius of ori computation.
             **********************************/
            radius = 1.5f * sigmas[i][j];
            for(ii = (int)radius+1; ii < h-(int)radius-1; ii++){
                for(jj = (int)radius+1; jj < w-(int)radius-1; jj++){
                    checkExtrema(self, 
                                 up, 
                                 down,
                                 w, h, 
                                 ii, jj,
                                 peakThres,
                                 &flagExtrema);
                    if(flagExtrema == 1){
                        checkEdge(dog[i][j],
                                  w, h,
                                  ii, jj,
                                  &flagEdge);
                        if(flagEdge == 1){
                            if(*keyN == 0){
                                // create head
                                (*keyHead) = 
                                    (pointList*)
                                    malloc
                                    (sizeof
                                     (pointList));
                                (*keyHead)->self.o =
                                    i;
                                (*keyHead)->self.s = 
                                    (float)j;
                                (*keyHead)->self.x = 
                                    (float)jj;
                                (*keyHead)->self.y = 
                                    (float)ii;
                                (*keyHead)->self.ori=
                                    0.0f;
                                (*keyHead)->next = 
                                    NULL;
                            }else{
                                // insert from head
                                pointList* tmp;
                                tmp =
                                    (pointList*)
                                    malloc(
                                        sizeof
                                        (pointList));
                                tmp->self.o = 
                                    i;
                                tmp->self.s = 
                                    (float)j;
                                tmp->self.x = 
                                    (float)jj;
                                tmp->self.y = 
                                    (float)ii;
                                tmp->self.ori = 
                                    0.0f;
                                tmp->next = 
                                    (*keyHead);
                                (*keyHead) = 
                                    tmp;
                            }
                            *keyN = *keyN+1;
                        }
                    }
                }
            }
        }
    }
}

/**********************************************
 * checkExtrema:
 * check whether a given pixel is extreme pt.
 *
 * Parameters
 * self: input, self scale in dog
 * up: input, upper scale in dog
 * down: input, lower scale in dog
 * w: intput, width of the octave
 * h: input, height of the octave
 * r: input, row (y) of the pt. to be checked
 * c: input, column (x) of the pt. to be checked
 * flagExtrema: output, whether (r,c) is extrema
 *********************************************/

void checkExtrema(float* self,
                  float* up,
                  float* down,
                  int w,
                  int h,
                  int r,
                  int c,
                  float peakThres,
                  int* flagExtrema){
    if( 
        ( self[r*w+c] < -1.0 * peakThres &&
          self[r*w+c] < self[(r-1)*w+c-1] &&
          self[r*w+c] < self[(r-1)*w+c] &&
          self[r*w+c] < self[(r-1)*w+c+1] &&
          self[r*w+c] < self[r*w+c-1] &&
          self[r*w+c] < self[r*w+c+1] &&
          self[r*w+c] < self[(r+1)*w+c-1] &&
          self[r*w+c] < self[(r+1)*w+c] &&
          self[r*w+c] < self[(r+1)*w+c+1] &&
          self[r*w+c] < up[(r-1)*w+c-1] &&
          self[r*w+c] < up[(r-1)*w+c] &&
          self[r*w+c] < up[(r-1)*w+c+1] &&
          self[r*w+c] < up[r*w+c-1] &&
          self[r*w+c] < up[r*w+c] &&
          self[r*w+c] < up[r*w+c+1] &&
          self[r*w+c] < up[(r+1)*w+c-1] &&
          self[r*w+c] < up[(r+1)*w+c] &&
          self[r*w+c] < up[(r+1)*w+c+1] &&
          self[r*w+c] < down[(r-1)*w+c-1] &&
          self[r*w+c] < down[(r-1)*w+c] &&
          self[r*w+c] < down[(r-1)*w+c+1] &&
          self[r*w+c] < down[r*w+c-1] &&
          self[r*w+c] < down[r*w+c] &&
          self[r*w+c] < down[r*w+c+1] &&
          self[r*w+c] < down[(r+1)*w+c-1] &&
          self[r*w+c] < down[(r+1)*w+c] &&
          self[r*w+c] < down[(r+1)*w+c+1] )
        ||
        ( self[r*w+c] > peakThres &&
          self[r*w+c] > self[(r-1)*w+c-1] &&
          self[r*w+c] > self[(r-1)*w+c] &&
          self[r*w+c] > self[(r-1)*w+c+1] &&
          self[r*w+c] > self[r*w+c-1] &&
          self[r*w+c] > self[r*w+c+1] &&
          self[r*w+c] > self[(r+1)*w+c-1] &&
          self[r*w+c] > self[(r+1)*w+c] &&
          self[r*w+c] > self[(r+1)*w+c+1] &&
          self[r*w+c] > up[(r-1)*w+c-1] &&
          self[r*w+c] > up[(r-1)*w+c] &&
          self[r*w+c] > up[(r-1)*w+c+1] &&
          self[r*w+c] > up[r*w+c-1] &&
          self[r*w+c] > up[r*w+c] &&
          self[r*w+c] > up[r*w+c+1] &&
          self[r*w+c] > up[(r+1)*w+c-1] &&
          self[r*w+c] > up[(r+1)*w+c] &&
          self[r*w+c] > up[(r+1)*w+c+1] &&
          self[r*w+c] > down[(r-1)*w+c-1] &&
          self[r*w+c] > down[(r-1)*w+c] &&
          self[r*w+c] > down[(r-1)*w+c+1] &&
          self[r*w+c] > down[r*w+c-1] &&
          self[r*w+c] > down[r*w+c] &&
          self[r*w+c] > down[r*w+c+1] &&
          self[r*w+c] > down[(r+1)*w+c-1] &&
          self[r*w+c] > down[(r+1)*w+c] &&
          self[r*w+c] > down[(r+1)*w+c+1] )
        ){
        *flagExtrema = 1;
    }else{
        *flagExtrema = 0;
    }
}

/**********************************************
 * checkEdge:
 * check whether a given pixel is edge
 *
 * Parameters
 * dogImg: input, diff of gaussian image
 * w: intput, width of the octave
 * h: input, height of the octave
 * r: input, row (y) of the pt. to be checked
 * c: input, column (x) of the pt. to be checked
 * flagEdge: output, whether (r,c) is edge
 *********************************************/

void checkEdge(float *dogImg, 
              int w,
              int h,
              int r,
              int c,
              int *flagEdge){


    float dxx, dyy, dxy;
    float center = dogImg[r*w+c];
    float left = dogImg[r*w+c-1];
    float right = dogImg[r*w+c+1];
    float up = dogImg[(r-1)*w+c];
    float down = dogImg[(r+1)*w+c];
    float upleft = dogImg[(r-1)*w+c-1];
    float upright = dogImg[(r-1)*w+c+1];
    float downleft = dogImg[(r+1)*w+c-1];
    float downright = dogImg[(r+1)*w+c+1];

    float trace, det, curv, K;

    // dirivertive with neighbor pixel
    dxx = right + left - 2.0f * center;
    dyy = up + down - 2.0f * center;
    // dirivertive with stride-of-two pixels
    // then averaged by two
    dxy = ( (upright - upleft)/2.0f -
            (downright - downleft)/2.0f ) / 2.0f;

    trace = dxx + dyy;
    det = dxx * dyy - dxy * dxy;
    curv = CURVATURE_THRESHOLD;
    K = curv / ((curv+1.0f)*(curv+1.0f));

    if(trace * trace / det < 1.0f / K){
        (*flagEdge) = 1;
    }else{
        (*flagEdge) = 0;
    }
}// checkEdge()
