#include "sift.h"
#include "timer.h"
#include "img_io.h"

#undef DUMP_OCTAVE
#undef DUMP_SCALESPACE
#undef DUMP_DOG

#ifdef _MSC_VER
// Visual C++ does not have log2...
float log2(float n) {  
    // log(n)/log(2) is log2.  
    return log( n ) / log( 2 );  
}
#endif

void
sift(unsigned char* img, int w, int h, char* out_filename) {
  
    /************************
     * Common Variables
     ***********************/
    Timer totalTime;
    int i, j;
#if (defined DUMP_OCTAVE || defined DUMP_SCALESPACE || defined DUMP_DOG)
    // file name
    char tmpfile[10];
#endif
    /************************
     * Global Variables
     ***********************/
    // number of total octaves
    int O;
    // number of scales per octaves
    int S;
    // width of each octave
    int octavesW[MAX_O];
    // height of each octave
    int octavesH[MAX_O];
    // sigma at each scale
    float sigma0 = 1.6f;
    float sigmas[MAX_O][MAX_S];
    // number of keys in linked list
    int keyN = 0;
    FILE *skey;
    /************************
     * Data Structures
     ***********************/

    // octaves generated by down sampling
    unsigned char* octaves[MAX_O];
    // scale space
    float* scaleSpace[MAX_O][MAX_S];
    // difference of gaussians
    float* dog[MAX_O][MAX_S-1];
    // extreme points
    pointList* keyHead, * keyPtr;

    /************************
     * Variable computation
     ***********************/
    // number of octaves (skip last two level)
    O = (int)(log2( (w < h) ? (float)w : (float)h))-2;
    // scales per octaves (epirical number)
    S = 6;
    // width of octaves at each level
    octavesW[0] = w;
    for(i = 1; i < O; i++){
        octavesW[i] = octavesW[i-1]/2;
    }
    // height of octaves at each level
    octavesH[0] = h;
    for(i = 1; i < O; i++){
        octavesH[i] = octavesH[i-1]/2;
    }
    // gaussian sigmas at each scale
    for(i = 0; i < O; i++){
        for(j = 0; j < S; j++){
            sigmas[i][j] = sigma0 * pow(sqrt(2.0f), (float)j/(float)(S-3) );
        }
    }

#if VERBOSE == 1
    printf("Image size: %dx%d\n", w, h);
    printf("Number of octaves (O):%d\n", O);
    printf("Scales per octave (S):%d\n", S);
    for(i = 0; i < O; i++){
        printf("size of octave level %d: %dx%d\n", i, octavesW[i], octavesH[i]);
    }
#endif

    /*******************************
     * start the timer
     *******************************/
    start_timer(&totalTime);
    /************************
     * Memory Allocation
     ***********************/
    // octaves
    for(i = 0; i < O; i++){
        octaves[i] = (unsigned char*)
            malloc(octavesW[i] * octavesH[i] * sizeof(unsigned char));
    }
    // scale sapce
    for(i = 0; i < O; i++){
        for(j = 0; j < S; j++){
            scaleSpace[i][j] =
                (float*)malloc(octavesW[i] * octavesH[i] * sizeof(float));
        }
    }
    // difference of gaussians
    for(i = 0; i < O; i++){
        for(j = 0; j < S-1; j++){
            dog[i][j] = 
                (float*)malloc( octavesW[i] * octavesH[i] * sizeof(float));
        }
    }
    
    /************************
     * Step 1:
     * Build Octave Base by downsampling
     ***********************/
    buildOB(img, octaves, O, S, octavesW, octavesH);
#ifdef DUMP_OCTAVE
    for(i = 0; i < O; i++){
        sprintf(tmpfile,"%s%d%s", "OB", i, ".pgm");
        write_pgm(octaves[i], octavesW[i], octavesH[i], tmpfile);
    }
#endif

    /************************
     * Step 2:
     * Build Scale Space by Gaussian
     ***********************/

    buildSS(octaves, scaleSpace, O, S, octavesW, octavesH, sigmas);

#ifdef DUMP_SCALESPACE
    for(i = 0; i < O; i++){
        for(j = 0; j < S; j++){
            sprintf(tmpfile, "%s%d%d%s", "SS", i, j, ".pgm");
            write_float_pgm(scaleSpace[i][j],octavesW[i],octavesH[i],tmpfile,1);
        }
    }
#endif

    /************************
     * Step 3:
     * Difference of Gaussians (DoG)
     ***********************/
    DoG(dog, scaleSpace, O, S, octavesW, octavesH);
#ifdef DUMP_DOG
    for(i = 0; i < O; i++){
        for(j = 0; j < S-1; j++){
            sprintf(tmpfile,"%s%d%d%s", "DoG", i, j, ".pgm");
            write_float_pgm(dog[i][j], octavesW[i], octavesH[i], tmpfile, 2);
        }
    }
#endif


    /************************
     * Step 4:
     * Find extreme points
     ***********************/
    extreme(&keyHead, &keyN, dog, O, S, octavesW, octavesH);

    /************************
     * Keypoint Orientation
     ***********************/
    orientation(keyHead, scaleSpace, sigmas, octavesW, octavesH);

#if VERBOSE == 1
    printf("Detect %d keypoints\n", keyN);
    skey = fopen("skey","w");
    keyPtr = keyHead;
    fprintf(skey, "%d\t%d\n", keyN, 4);
    for(i = 0; i < keyN; i++){
        fprintf(skey, "%f\t%f\t%f\t%f\n", keyPtr->self.x, keyPtr->self.y,
                keyPtr->self.s, keyPtr->self.ori);
        keyPtr = keyPtr->next;
    }
    fclose(skey);
    printf
        ("Output 4 element descriptor to file: skey\n");
#endif

    /*******************************
     * All actual processing done,
     * stop the timer
     *******************************/
    stop_timer(&totalTime);
    printf("Total time: %f ms\n", get_timer_interval(&totalTime));
    /************************
     * Render Results
     * Draw circle around keypoints
     * (to do: draw the orientation)
     ***********************/
    render(img, out_filename, O, S, octavesW, octavesH, keyHead, keyN, sigmas);

    /************************
     * Free Memory
     ***********************/
    for(i = 0; i < O; i++){
        free(octaves[i]);
        for(j = 0; j < S; j++){
            free(scaleSpace[i][j]);
            if(j < S-1){
                free(dog[i][j]);
            }
        }
    }
}// sift()
